#include "SSH.h"

namespace Upp {

#define LLOG(x)	 	RLOG(x)

void SFtp::StartInit()
{
	AddJob() << [=] {
		ASSERT(ssh);
		auto ss = ssh->GetSession();
		if(!ss)
			Error(-1, t_("Unable to initialize Sftp. The ssh session is invalid."));
		session = libssh2_sftp_init(ss);
		if(session) {
			LLOG("++ SFTP: Session succesfully started.");
			return false;
		}
		if(!WouldBlock()) Error();
		return true;
	};
}

void SFtp::StartStop()
{
	AddJob() << [=] {
		if(!session)
			// Since we have yet to properly clean up the ssh session, we do not halt the queue.
			return false;
		auto rc = libssh2_sftp_shutdown(session);
		if(rc == 0) {
			LLOG("++ SFTP: Session successfully stopped.");
			session = NULL;
			return false;
		}
		// For some reason, libssh2 does not report shutdown errors to libssh2_session_last_errno().
		if(rc != LIBSSH2_ERROR_EAGAIN) Error();
		return true;
	};
}

void SFtp::StartFStat(SFtpAttrs& attrs, bool set)
{
	auto *p = &attrs;
	AddJob() << [=] {
		auto rc = libssh2_sftp_fstat_ex(handle, p, set);
		if(rc == 0){
			LLOG("++ SFTP: File attributes succesfully retrieved.");
			return false;
		}
		if(rc != LIBSSH2_ERROR_EAGAIN) Error();
		return true;
	};
}

void SFtp::StartSymLink(const String& path, String* target, int type)
{
	if(type == LIBSSH2_SFTP_SYMLINK)
		AddJob() << [=] {
			ASSERT(session);
			int rc = libssh2_sftp_symlink_ex(
						session,
						path,
						path.GetLength(),
						const_cast<char*>(target->Begin()),
						target->GetLength(),
						type
					);
			if(rc == 0) {
				LLOG("++ SFTP: Symbolic link [\"" + *target + "\"]  is successfully created.");
				return false;
			}
			if(rc != LIBSSH2_ERROR_EAGAIN) Error();
			return true;
		};
	else
		AddJob() << [=] {
			ASSERT(session);
			Buffer<char> buf(512, 0);
			int rc = libssh2_sftp_symlink_ex(
						session,
						path,
						path.GetLength(),
						buf,
						512,
						type
					);
			if(rc > 0) {
				LLOG("++ SFTP: Symbolic link operation is successful.");
				target->Set(buf, rc);
				return false;
			}
			if(rc != LIBSSH2_ERROR_EAGAIN) Error();
			return true;
		};
}

SFtp& SFtp::StartOpen(const String& path, unsigned long flags, long mode)
{
	AddJob() << [=] {
		ASSERT(session);
		if((handle = libssh2_sftp_open(session, path, flags, mode))) {
			LLOG("++ SFTP: Path [\"" + path + "\"] successfully opened.");
			return false;
		}
		if(!WouldBlock()) Error();
		return true;
	};
	return *this;
}

SFtp& SFtp::StartClose()
{
	AddJob() << [=] {
		ASSERT(session);
		if(!handle) {
			LLOG("-- SFTP: No file handle to close.");
			return false;
		}
		auto rc = libssh2_sftp_close_handle(handle);
		if(rc == 0) {
			LLOG("++ SFTP: Path succesfully closed.");
			handle = NULL;
			return false;
		}
		if(rc != LIBSSH2_ERROR_EAGAIN) Error();
		return true;
	};
	return *this;
}

SFtp& SFtp::StartRemove(const String& path)
{
	AddJob() << [=] {
		ASSERT(session);
		auto rc = libssh2_sftp_unlink(session, path);
		if(rc == 0) {
			LLOG("++SFTP: File [\"" + path + "\"] successfully deleted.");
			return false;
		}
		if(rc != LIBSSH2_ERROR_EAGAIN) Error();
		return true;
	};
	return *this;
}

SFtp& SFtp::StartRename(const String& oldpath, const String& newpath)
{
	AddJob() << [=] {
		ASSERT(session);
		auto rc = libssh2_sftp_rename(session, oldpath, newpath);
		if(rc == 0) {
			LLOG("++ SFTP: File/directory succesfully renamed or moved.");
			return false;
		}
		if(rc != LIBSSH2_ERROR_EAGAIN) Error();
		return true;
	};
	return *this;
}

SFtp& SFtp::StartSync()
{
	AddJob() << [=] {
		ASSERT(session);
		auto rc = libssh2_sftp_fsync(handle);
		if(rc == 0) {
			LLOG("++SFTP: File successfully synchronized to disk.");
			return false;
		}
		if(rc != LIBSSH2_ERROR_EAGAIN) Error();
		return true;
	};
	return *this;
}

SFtp& SFtp::StartOpenDir(const String& path)
{
	AddJob() << [=] {
		ASSERT(session);
		auto *hndl = libssh2_sftp_opendir(session, path);
		if(hndl) {
			LLOG("++ SFTP: Directory [\"" + path + "\"] successfully opened.");
			handle = hndl;
			return  false;
		}
		if(!WouldBlock()) Error();
		return true;
	};
	return *this;
}

SFtp& SFtp::StartListDir(DirList& list, Gate<DirEntry&> progress)
{
	AddJob() << [=, &list] {
		ASSERT(handle);
		char label[512];
		char longentry[512];
		SFtpAttrs attrs;
		int rc = libssh2_sftp_readdir_ex(handle, label, sizeof(label), longentry, sizeof(longentry), &attrs);
		if(rc == 0)
			return false;
		else
		if(rc > 0) {
//			LLOG("|| " << longentry);
			SFtp::DirEntry& entry	= list.Add();
			entry.file 		 		= label;
			entry.entry 			= longentry;
			entry.a.flags			= attrs.flags;
			entry.a.permissions 	= attrs.permissions;
			entry.a.uid				= attrs.uid;
			entry.a.gid				= attrs.gid;
			entry.a.atime			= attrs.atime;
			entry.a.mtime			= attrs.mtime;
			entry.a.filesize		= attrs.filesize;
			progress(entry);
			return true;
		}
		if(rc != LIBSSH2_ERROR_EAGAIN) Error();
		return true;
	};
	return *this;
}

SFtp& SFtp::StartListDir(const String& path, DirList& list, Gate<DirEntry&> progress)
{
	StartOpenDir(path);
	StartRealPath(path, dir);
	StartListDir(list, pick(progress));
	StartClose();
}

SFtp& SFtp::StartMakeDir(const String& path, long mode)
{
	AddJob() << [=] {
		ASSERT(session);
		auto rc = libssh2_sftp_mkdir(session, path, mode);
		if(rc == 0) {
			LLOG("++ SFTP: Directory succesfully created.");
			return false;
		}
		if(rc != LIBSSH2_ERROR_EAGAIN) Error();
		return true;
	};
	return *this;
}

SFtp& SFtp::StartRemoveDir(const String& path)
{
	AddJob() << [=] {
		ASSERT(session);
		auto rc = libssh2_sftp_rmdir(session, path);
		if(rc == 0) {
			LLOG("++ SFTP: Directory succesfully removed.");
			return false;
		}
		if(rc != LIBSSH2_ERROR_EAGAIN) Error();
		return true;
	};
	return *this;
}

SFtp& SFtp::StartPut(const Stream& in, Gate<int64, int64> progress)
{
	packet_length = 0;
	AddJob() << [=, &in] {
		ASSERT(handle);
		int64 remaining = in.GetSize() - packet_length;
		int rc = libssh2_sftp_write(handle, const_cast<Stream&>(in).Get(remaining), remaining);
		if(rc < 0) {
			if(rc != LIBSSH2_ERROR_EAGAIN) Error();
			return true;
		}
		packet_length += rc;
		if(progress(in.GetSize(), packet_length))
			Error(-1, t_("File upload aborted."));
		if(rc == 0 & remaining == 0) {
			LLOG(Format("++ SFTP: %ld of %ld bytes successfully written.",	packet_length, in.GetSize()));
			packet_length = 0;
			return false;
		}
		return true;
	};
	return *this;
}

SFtp& SFtp::StartPut(const Stream& in, const String& path, unsigned long flags, long mode, Gate<int64, int64> progress)
{
	StartOpen(path, flags, mode);
	StartPut(in, progress);
	return StartClose();
}

SFtp& SFtp::StartGet(Stream& out, Gate<int64, int64> progress)
{
	packet_length = 0;
	AddJob() << [=, &out] {
		ASSERT(handle);
		Buffer<char> buffer(chunk_size);
		int rc = libssh2_sftp_read(handle, buffer, chunk_size);
		if(rc > 0) {
			out.Put(buffer, rc);
			if(progress(dir_entry.GetSize(), out.GetSize()))
				Error(-1, t_("File download aborted."));
			return true;
		}
		else
		if(rc == 0) {
			LLOG(Format("++ SFTP: %ld of %ld bytes successfully read.",	out.GetSize(), dir_entry.GetSize()));
			return false;
		}
		if(rc != LIBSSH2_ERROR_EAGAIN) Error();
		return true;
	};
	return *this;
}

SFtp& SFtp::StartGet(Stream& out, const String& path, unsigned long flags, long mode, Gate<int64, int64> progress)
{
	StartOpen(path, flags, mode);
	StartGetStat(dir_entry.GetAttrs());
	StartGet(out, progress);
	return StartClose();
}

bool SFtp::Seek(int64 position)
{
	if(!handle) {
		LLOG(("-- SFTP: Seek failed. Invalid file handle."));
		return false;
	}
	libssh2_sftp_seek64(handle, position);
	return true;
}

int64 SFtp::Tell()
{
	if(!handle) {
		LLOG(("-- SFTP: Tell failed. Invalid file handle."));
		return -1;
	}
	return libssh2_sftp_tell64(handle);
}

void SFtp::CleanUp()
{
	if(handle) StartClose();
	if(session) StartStop();
	if(!IsCleanup()) Execute();
	else LLOG("** SFTP: Perfoming clean up...");
}

SFtp::SFtp()
: SshSubsystem()
{
	type = Type::SFTP;
	session = NULL;
	handle = NULL;
}

bool SFtp::DirEntry::CanMode(dword u, dword g, dword o) const
{
	return a.flags & LIBSSH2_SFTP_ATTR_PERMISSIONS &&
		   a.permissions & o ||
		   a.permissions & g ||
		   a.permissions & u;
}
}